//****************************************************************************************
// Name:		proc.cpp
// Platform:	SQL Server 2000 SP3a or higher, Windows NT, 2000 or XP
// Author:		Copyright (c) 2006 by Michael Coles, MCDBA
//
// Description:	This code implements the xp_hash_sha extended stored procedure.  This 
//				code is built upon the SHA2 hash generator by Dr Brian Gladman.  See
//				sha2.cpp file for copyright information from Dr. Gladman.
//
// LEGAL STUFF:
// ------------
// Copyright (C) 2005 - 2006 by Michael Coles, MCDBA
//
// Some included code included is released under the redistribution agreements as 
// specified by the authors of the respective code.  Copyright holders of this included 
// code maintain all copyright and other rights to their original or derivative works.
//
// All rights reserved.                          
//
// REDISTRIBUTION OF THIS CODE:
// ----------------------------
// All code included in this package is either the original work of the copyright holder,
// or derivative work based on other copyright holders' works.  All derivative works 
// include information as required by the copright holders' redistribution agreements.
// These redistribution agreements, where possible, are included in the text of the source
// code distributed with this code.
//
// Redistribution and use in source and binary forms, with or without modification, are 
// permitted provided that the following conditions are met:
//
//   1. Redistributions of source code must retain the above copyright notice, this list 
//      of conditions and the following disclaimer.
//
//   2. Redistributions in binary form must reproduce the above copyright notice, this 
//      list of conditions and the following disclaimer in the documentation and/or other 
//      materials provided with the distribution.
//
//   3. The names of its contributors may not be used to endorse or promote products 
//      derived from this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY 
// EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES 
// OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT 
// SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
// INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED 
// TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR 
// BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
// CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY 
// WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//****************************************************************************************
#include <dblib.h>
#include <params.h>
#include <constants.h>
#include <sha2.h>
#include <stdafx.h>
#include <stdexcept>

#ifdef __cplusplus
extern "C" {
#endif

RETCODE __declspec(dllexport) xp_hash_sha(SRV_PROC *srvproc);

#ifdef __cplusplus
}
#endif

RETCODE __declspec(dllexport) xp_hash_sha(SRV_PROC *srvproc)
{
	RETCODE rc = XP_NOERROR;
	params *P1 = new params();	// Plain text
	params *P2 = new params();	// Hash code OUTPUT
	params *P3 = new params();	// Bits
	BYTE *plaintext = NULL;
	ULONG plaintextlength = 0;
	BYTE *hashcode = NULL;
	ULONG hashcodelength = 0;
	try 
	{
		int numparams = params::getparamcount(srvproc);
		if (numparams != 3) {
			Dblib::printerror(srvproc, USAGE_HASH_SHA);
			rc = XP_ERROR;
		} else {
			params::getparam(srvproc, 1, P1);
			params::getparam(srvproc, 2, P2);
			params::getparam(srvproc, 3, P3);
			if (P1->isoutput || !P2->isoutput || P3->isoutput) {
				Dblib::printerror(srvproc, USAGE_HASH_SHA);
				rc = XP_ERROR;
			} else {
				if (P1->isnull) {
					rc = XP_ERROR;
				}
			}
		}
		if (rc == XP_NOERROR) {
			if (P3->isnull) {
				hashcodelength = SHA512_DIGEST_LENGTH;
			} else {
				ULONG *i = ((ULONG *)P3->cdata);
				switch (*i) {
					case 512:
						hashcodelength = SHA512_DIGEST_LENGTH;
						break;
					case 384:
						hashcodelength = SHA384_DIGEST_LENGTH;
						break;
					case 256:
						hashcodelength = SHA256_DIGEST_LENGTH;
						break;
					default:
						Dblib::printerror(srvproc, ERR_BAD_HASH_BITS);
						rc = XP_ERROR;
						break;
				}
			}
		}
		if (rc == XP_NOERROR) {
			hashcode = new BYTE[hashcodelength];
			plaintext = new BYTE[P1->length];
			memcpy(plaintext, P1->cdata, P1->length);
			plaintextlength = P1->length;
			switch (hashcodelength) {
				case SHA512_DIGEST_LENGTH:
					sha512_ctx ctx1[1];
					sha512_begin (ctx1);
					sha512_hash(plaintext, plaintextlength, ctx1);
					sha512_end(hashcode, ctx1);
					break;
				case SHA384_DIGEST_LENGTH:
					sha384_ctx ctx2[1];
					sha384_begin (ctx2);
					sha384_hash(plaintext, plaintextlength, ctx2);
					sha384_end(hashcode, ctx2);
					break;
				case SHA256_DIGEST_LENGTH:
					sha256_ctx ctx3[1];
					sha256_begin (ctx3);
					sha256_hash(plaintext, plaintextlength, ctx3);
					sha256_end(hashcode, ctx3);
					break;
				default:
					rc = XP_ERROR;
					break;
			}
		}
	} catch(std::exception ex) {
		Dblib::printerror(srvproc, ERR_HASH_EXCEPTION);
		rc = XP_ERROR;
	}
	if (rc == XP_NOERROR) {
		srv_paramsetoutput(srvproc, 2, hashcode, hashcodelength, FALSE);
	} else {
		srv_paramsetoutput(srvproc, 2, (BYTE *)"", 0, TRUE);
	}
	srv_senddone(srvproc, 0, 0, 0);
	if (plaintext != NULL) {
		srv_bzero(plaintext, plaintextlength);
		delete [] plaintext;
	}
	plaintext = NULL;
	if (hashcode != NULL) {
		srv_bzero(hashcode, hashcodelength);
		delete [] hashcode;
	}
	hashcode = NULL;
	if (P1 != NULL)
		delete P1;
	if (P2 != NULL)
		delete P2;
	if (P3 != NULL)
		delete P3;
	return rc;
}

